﻿/**********************************************************************
// Copyright 2018 Autodesk, Inc.  All rights reserved. 
**********************************************************************/
---------------------------------------------------------
-- This block is for future auto-discovery of security tools. 
-- The following is the global variable name of the singleton instance created by running this script.
--  security_tool_instance_name = "ALC_SecurityTool"
---------------------------------------------------------

---------------------------------------------------------
-- The only method required to be implemented as a public method is:
--   'disable()' - disables the tool and sets the global variable name to undefined
-- The disable method is called by SecurityToolsStartup_instance.display_permission_dialog when disabling protection
-- The tool would be re-enabled by re-running the script.
---------------------------------------------------------

-- ALC persistent global variable and callback removal, cleaning of ALC scripted controllers, removal of ALC startup script files
-- version 1.0 - initial version of script in package file

-- load structure that handles dialogs
if ::SecurityToolsDialogs_instance == undefined do
	filein (pathconfig.appendPath (pathconfig.removePathLeaf (getThisScriptFilename())) "SecurityToolsDialogs.ms")
	
struct ALC_SecurityTool_structdef 
(
	private
	m_callbacks_show_support_to = false, -- whether callbacks.show supports to:<stream>
	m_problem_found_count = 0, -- incremented each time any sort of ALC problem (startup scripts or scene) is found.
	m_problem_fixed_count = 0, -- incremented each time any sort of ALC cleanup (startup scripts or scene) occurs.
	m_perform_cleanup_quiet_mode_default = true, -- default cleanp behavior when in quiet mode
	m_security_tools_ALC_help_topic_id = "idh_security_tools_alc",
	m_ALC_startup_script_files_found, -- the ALC startup script files found by test_for_ALC_startup_scripts
	m_perform_ALC_startup_script_file_cleanup, -- whether to clean up the ALC startup script files
	m_perform_ALC_process_cleanup, -- whether to clean up ALC corrupted scene files
	m_postloadCallbacks_registered = false, -- used for controlling registration of postload callbacks
	m_force_local_quiet_mode= false, -- if true, behave as if in quiet mode
		
	m_perform_cleanup_quiet_mode = m_perform_cleanup_quiet_mode_default,-- cleanup behavior when in quiet mode
	
	m_in_objectXrefMerge = false, -- used for controlling register of #objectXrefPostMerge
	-- The following is for remembering the response when finding scene file corruption via test_for_ALC_scripted_controller_callback().  The test_for_ALC_scripted_controller_callback() call will be 
	-- followed later in the load process with a call to test_for_ALC_process_corruption(). We want the reponse in test_for_ALC_scripted_controller_callback() to be applied automatically to the 
	-- test_for_ALC_process_corruption() call. It is the same scene being loaded, and we are saying whether or not we want that scene cleaned. Only 1 response is needed.
	-- Note that the test_for_ALC_scripted_controller_callback() call only occurs when doing merges and loading object xrefs. During normal scene loads, the #nodeLinked event does not occur.
	m_auto_accept_next_process_clean = false, -- test_for_ALC_scripted_controller_callback finds scripted controllers before nodes are created, if clean scripts also want to clean up nodes 
	m_auto_reject_next_process_clean = false, -- test_for_ALC_scripted_controller_callback finds scripted controllers before nodes are created, if don't clean scripts also don't want to clean up nodes 
	m_in_objectXrefMerge_depth = 0,
	m_in_sceneXrefMerge = false, -- used for controlling register of #objectXrefPostMerge
	m_in_sceneXrefMerge_depth = 0, 
	m_nodelinked_callback_registered = false, -- used for controlling register of #nodeLinked
	
	m_debug_mode_on = false, -- enables debugging output

	-- things to look for....
	m_ALC_script_files = #(@"$userStartupScripts\vrdematcleanbeta.ms", @"$userStartupScripts\vrdematcleanbeta.mse", @"$userStartupScripts\vrdematcleanbeta.msex"),
	m_ALC_global_to_search = #AutodeskLicSerStuckCleanBeta,
	m_ALC_globals_to_clean = #(#physXCrtRbkInfoCleanBeta, #checkLicSerSubCleanBeta, #checkLicSerMainCleanBeta, #CleanBetabaseCC64enc, #CleanBetabaseCC64dec, #runMainCleanBeta, #PointNodeCleanBeta, 
			#px_HiddenNodeCleanBeta, #getNetUpdateCleanBeta, #AutodeskLicSerStuckCleanBeta, #px_SimulatorForModifyCleanBeta, #px_SimulatorForStateCleanBeta, #px_SimulatorSaveCleanBeta),
	m_ALC_callback_ids = #(#RenderLicCleanBeta,#PhysXCleanBetaRBKSysInfo,#AutodeskLicCleanBeta),
	m_ALC_node_names = #("×þ×ü", "¡¡×ý×û"),
	m_scale_script_expression_string = "AutodeskLicSerStuckCleanBeta()",
			
	m_version_number = "1.0",

	public
			
	m_in_file_open = false, -- used for controlling register of filePostMerge
	m_force_print_to_listener = false, -- whether to display listener output when not in quiet mode

	-- disable this tool - required  method
	fn disable =
	(
		-- unregister callbacks
		callbacks.removeScripts id:#ALC_SecurityTool
		callbacks.removeScripts id:#ALC_SecurityTool_startup
		callbacks.removeScripts id:#ALC_SecurityTool_postload
		callbacks.removeScripts id:#ALC_SecurityTool_postload_nodelinked
		m_postloadCallbacks_registered = false
		m_nodelinked_callback_registered = false
		-- log info
		local msg = ~ALC_SECURITY_TOOL~ + m_version_number + ~ALC_SECURITY_TOOLS_DISABLED~
		logsystem.logEntry msg info:true broadcast:true
		if m_force_local_quiet_mode or GetQuietMode() or m_force_print_to_listener do
			format "-- %\n" msg
		-- set global instance variable to undefined
		 ::ALC_SecurityTool = undefined
	),

	-- get the problem found and fixed counts. Used by unit tests
	fn get_problem_found_count = return m_problem_found_count,
	fn get_problem_fixed_count = return m_problem_fixed_count,

	-- reset problem found and fixed counts. Used by unit tests
	fn clear_problem_counts = m_problem_found_count = m_problem_fixed_count = 0,

	-- get the name of this script file. Used by unit tests
	fn getToolFileName =
	(
		return (::getSourceFileName())
	),
	
	-- enable debugging
	fn enable_Debug = 
	(
		m_debug_mode_on = true
		callbacks.removeScripts id:#ALC_SecurityTool_Debugging
		callbacks.addScript #filePreOpen "print #filePreOpen" id:#ALC_SecurityTool_Debugging
		callbacks.addScript #filePostOpen "print #filePostOpen" id:#ALC_SecurityTool_Debugging
		callbacks.addScript #fileOpenFailed "print #fileOpenFailed" id:#ALC_SecurityTool_Debugging
		callbacks.addScript #filePreMerge "print #filePreMerge" id:#ALC_SecurityTool_Debugging
		callbacks.addScript #filePostMerge "print #filePostMerge" id:#ALC_SecurityTool_Debugging
		callbacks.addScript #filePostMerge2 "print #filePostMerge2" id:#ALC_SecurityTool_Debugging
		callbacks.addScript #sceneXrefPreMerge "print #sceneXrefPreMerge" id:#ALC_SecurityTool_Debugging
		callbacks.addScript #sceneXrefPostMerge "print #sceneXrefPostMerge" id:#ALC_SecurityTool_Debugging
		callbacks.addScript #objectXrefPreMerge "print #objectXrefPreMerge" id:#ALC_SecurityTool_Debugging
		callbacks.addScript #objectXrefPostMerge "print #objectXrefPostMerge" id:#ALC_SecurityTool_Debugging
		callbacks.addScript #filePostOpenProcessFinalized "print #filePostOpenProcessFinalized" id:#ALC_SecurityTool_Debugging
		callbacks.addScript #filePostMergeProcessFinalized "print #filePostMergeProcessFinalized" id:#ALC_SecurityTool_Debugging
		callbacks.addScript #nodeLinked "print #nodeLinked" id:#ALC_SecurityTool_Debugging
		callbacks.addScript #animationRangeChange "print #animationRangeChange" id:#ALC_SecurityTool_Debugging
	),
	
	-- disable debugging
	fn disable_Debug = 
	(
		m_debug_mode_on = false
		callbacks.removeScripts id:#ALC_SecurityTool_Debugging
	),
	
	-- for debugging, testing
	fn get_auto_accept_values = 
	(
		datapair m_auto_accept_next_process_clean m_auto_reject_next_process_clean
	),

	-- test environment for ALC
	fn test_for_ALC_startup_scripts =
	(
		m_ALC_startup_script_files_found = #()
		for fname in m_ALC_script_files do
		(
			if doesFileExist fname do
				append m_ALC_startup_script_files_found (pathconfig.resolvePathSymbols fname)
		)
		
		if (m_ALC_startup_script_files_found.count != 0) do
		(
			m_problem_found_count += 1
			local msg = ~ALC_SECURITY_TOOL~ + m_version_number + ~FOUND_ALC_STARTUP_SCRIPT_FILE~
			for fname in m_ALC_startup_script_files_found do
				logsystem.logentry (msg + fname) warning:true broadcast:true
			
			local display_listener_output = m_force_local_quiet_mode or GetQuietMode() or m_force_print_to_listener

			if display_listener_output do
			(
				msg = ~ALC_SECURITY_TOOL~ + m_version_number + ~FOUND_FOLLOWING_ALC_STARTUP_SCRIPT_FILES~
				msg += ~FILES_MAY_HAVE_HIDDEN_AND_SYSTEM_FILE_ATTRIBUTES_SET~ + "\n"
				for f in m_ALC_startup_script_files_found do
					msg += ("  " + f + "\n")
				format "%" msg
			)
			
			m_perform_ALC_startup_script_file_cleanup = m_perform_cleanup_quiet_mode 
			if not (m_force_local_quiet_mode or GetQuietMode()) then
			(
				m_perform_ALC_startup_script_file_cleanup = SecurityToolsDialogs_instance.display_corruptionFoundInEnv_dialog m_ALC_startup_script_files_found m_security_tools_ALC_help_topic_id "ALC"
				if not m_perform_ALC_startup_script_file_cleanup do
				(
					SecurityToolsDialogs_instance.display_corruptionNotCleanedInEnv_dialog m_security_tools_ALC_help_topic_id
				)
			)
			else if display_listener_output do
			(
				if m_perform_ALC_startup_script_file_cleanup then
					msg = ~ALC_ENV_AUTO_CLEAN_QUIET~ + "\n"
				else
					msg = ~ALC_ENV_NOT_AUTO_CLEAN_QUIET~ + "\n"
				format "%" msg
			)
			
			if m_perform_ALC_startup_script_file_cleanup then
			(
				m_problem_fixed_count += 1
				
				-- remove the files
				local undeleted_files = #()
				for fname in m_ALC_startup_script_files_found do
				(
					if (deleteFile fname) then
					(
						msg = ~ALC_SECURITY_TOOL~ + m_version_number + ~ALC_ENV_DELETED_FILE~ + fname
						logsystem.logentry msg info:true broadcast:true
						if display_listener_output do
							format "%\n" msg
					)
					else
					(
						msg = ~ALC_SECURITY_TOOL~ + m_version_number + ~ALC_ENV_ERROR_DELETING_FILE~ + fname
						logsystem.logentry msg warning:true broadcast:true
						if display_listener_output do
							format "%\n" msg
						append undeleted_files fname
					)
				)
				if undeleted_files.count == 0 then
				(
					msg = ~ALC_SECURITY_TOOL~ + m_version_number + " - " + ~ALC_ENV_CLEAN_SUCCESS~
					logsystem.logentry msg info:true broadcast:true
					if display_listener_output do
						format "%\n" msg
					if securitytools != undefined do
						securitytools.LogEventById #EnvironmentALC #CleanupSucceeded
					
					if not (m_force_local_quiet_mode or GetQuietMode()) then
						SecurityToolsDialogs_instance.display_corruptionCleanedInEnv_dialog m_security_tools_ALC_help_topic_id
				)
				else
				(
					msg = ~ALC_SECURITY_TOOL~ + m_version_number + ~ALC_ENV_CLEAN_FAILURE~
					logsystem.logentry msg warning:true broadcast:true
					if securitytools != undefined do
						securitytools.LogEventById #EnvironmentALC #CleanupFailed
					
					if m_force_local_quiet_mode or GetQuietMode() then
					(
						local ss = stringstream ""
						format (~ALC_ENV_CLEAN_FAILURE_FMT~ + "\n") (m_ALC_startup_script_files_found.count - undeleted_files.count) m_ALC_startup_script_files_found.count to:ss
						msg = ss as string
						for f in undeleted_files do
							msg += ("  " + f + "\n")
						msg += ~ALC_ENV_CLEAN_FAILURE_MSG~ + "\n"
						msg += m_security_tools_ALC_help_topic_id
						format "%" msg
					)
					else
					(
						SecurityToolsDialogs_instance.display_corruptionErrorCleaningInEnv_dialog undeleted_files m_security_tools_ALC_help_topic_id
					)
				)
			)
			else
			(
				if (m_ALC_startup_script_files_found.count != 0) do
				(
					if securitytools != undefined do
						securitytools.LogEventById #EnvironmentALC #CleanupDeclined
					for fname in m_ALC_startup_script_files_found do
					(
						local msg = ~ALC_SECURITY_TOOL~ + m_version_number + ~ALC_ENV_CLEAN_FAILURE_MSG2~ + fname
						logsystem.logentry msg warning:true broadcast:true
						if display_listener_output do 
							format "%\n" msg
					)
				)
			)
		)
	),
	
	-- test process for ALC corruption
	fn test_for_ALC_process_corruption unregisterPostLoad:false unregisterPostStartup:false useSafeNodeNameCollect:false doingPreSaveCheck:false saveFileName:"" =
	(
		if m_debug_mode_on do
		(
			format "test_for_ALC_process_corruption unregisterPostLoad:% unregisterPostStartup:% useSafeNodeNameCollect:% doingPreSaveCheck:% saveFileName:%\n" unregisterPostLoad unregisterPostStartup useSafeNodeNameCollect doingPreSaveCheck saveFileName
			format "test_for_ALC_process_corruption m_in_file_open:% m_in_objectXrefMerge:% m_in_objectXrefMerge_depth:% m_in_sceneXrefMerge:% m_in_sceneXrefMerge_depth:% m_auto_accept_next_process_clean:% m_auto_reject_next_process_clean:%\n" m_in_file_open m_in_objectXrefMerge m_in_objectXrefMerge_depth m_in_sceneXrefMerge m_in_sceneXrefMerge_depth m_auto_accept_next_process_clean m_auto_reject_next_process_clean
		)
		
		if unregisterPostLoad do
		(
			callbacks.removeScripts id:#ALC_SecurityTool_postload
			callbacks.removeScripts id:#ALC_SecurityTool_postload_nodelinked
			callbacks.removeScripts id:#ALC_SecurityTool_animationRangeChange
			m_postloadCallbacks_registered = false
			m_nodelinked_callback_registered = false
		)
		if unregisterPostStartup do
			callbacks.removeScripts id:#ALC_SecurityTool_startup
		
		-- test for ALC globals
		local found_ALC_global = globalVars.isGlobal m_ALC_global_to_search
		if found_ALC_global do
			m_problem_found_count += 1
		
		-- test for ALC callbacks.
		local found_ALC_callbacks = false
		-- This test is slow - only do if found ALC globals
		if found_ALC_global do
		(
			if m_callbacks_show_support_to then
			(
				local ss = stringstream ""
				for id in m_ALC_callback_ids do
					callbacks.show id:id to:ss
				seek ss 0
				while not eof ss and not found_ALC_callbacks do
				(
					local l = readline ss
					if l != "OK" do found_ALC_callbacks = true
				)
				free ss
			)
			else
			(
				-- just assume that the callbacks are present, otherwise we spam listener
				found_ALC_callbacks = true
			)

			if found_ALC_callbacks do
				m_problem_found_count += 1
		)
		
		-- test for ALC script controllers
		local scaleScripts = getclassinstances scale_script processAllAnimatables:true
		local the_ALC_script_controllers = #()
		local found_ALC_script_controllers = false
		for scaleScript in scaleScripts do 
		(
			local expr = scaleScript.GetExpression()
			if (findstring expr m_scale_script_expression_string) != undefined do
				append the_ALC_script_controllers scaleScript
		)
		if the_ALC_script_controllers.count != 0 do
		(
			found_ALC_script_controllers = true
			m_problem_found_count += 1
		)		
		
		m_perform_ALC_process_cleanup = false
		local process_corruption_found = found_ALC_global or found_ALC_callbacks or found_ALC_script_controllers
		
		if process_corruption_found do
		(
			local scene_file_name
			if doingPreSaveCheck then
				scene_file_name = saveFileName
			else
				scene_file_name = maxFilePath + maxFileName
			if scene_file_name == "" do scene_file_name = ~ALC_SCENE_UNNAMED~
			
			local display_listener_output = m_force_local_quiet_mode or GetQuietMode() or m_force_print_to_listener

			local msg
			-- if m_auto_accept_next_process_clean or m_auto_reject_next_process_clean is true, no messaging - using response from finding corruption in test_for_ALC_scripted_controller_callback
			if not (m_auto_accept_next_process_clean or m_auto_reject_next_process_clean) do
			(
				if doingPreSaveCheck then
					msg = ~ALC_SECURITY_TOOL~ + m_version_number + ~ALC_SCENE_DETECTED_MSG_PRESAVE~ + scene_file_name
				else
					msg = ~ALC_SECURITY_TOOL~ + m_version_number + ~ALC_SCENE_DETECTED_MSG~ + scene_file_name
				logsystem.logentry msg warning:true broadcast:true
				if display_listener_output do
					format "%\n" msg
			)

			if m_auto_accept_next_process_clean then
				m_perform_ALC_process_cleanup = true
			else if m_auto_reject_next_process_clean then
				m_perform_ALC_process_cleanup = false
			else if m_force_local_quiet_mode or GetQuietMode() then
			(
				m_perform_ALC_process_cleanup = m_perform_cleanup_quiet_mode
				if m_perform_ALC_process_cleanup then
				(
					msg = ~ALC_SECURITY_TOOL~ + m_version_number + ~ALC_SCENE_AUTO_CLEAN_QUIET~
				)
				else
				(
					msg = ~ALC_SECURITY_TOOL~ + m_version_number + ~ALC_SCENE_NOT_AUTO_CLEAN_QUIET~
				)
				logsystem.logentry msg info:true broadcast:true
				format "%\n" msg
			)
			else
			(
				-- even though we disable viewport redraw in the dialog open handler, there is at least one code path that doesn't check whether viewport redraws are disabled
				-- before redrawing the viewports. While dialog is up, replace the script controllers in the scene with different controllers, and then restore.
				local controller_remappings
				if found_ALC_script_controllers do with undo off
				(
					controller_remappings = #()
					for c in the_ALC_script_controllers do
					(
						local new_c = bezier_scale()
						replaceinstances c new_c
						append controller_remappings (datapair c new_c)
					)
				)
				m_perform_ALC_process_cleanup = SecurityToolsDialogs_instance.display_corruptionFoundInScene_dialog m_security_tools_ALC_help_topic_id "ALC"
				if found_ALC_script_controllers do with undo off
				(
					for item in controller_remappings do
						replaceInstances item.v2 item.v1
				)
				if not m_perform_ALC_process_cleanup do
				(
					SecurityToolsDialogs_instance.display_corruptionNotCleanedInScene_dialog m_security_tools_ALC_help_topic_id
				)
			)
			
			if m_perform_ALC_process_cleanup then
			(
				-- remove the callbacks
				if found_ALC_callbacks do
				(
					m_problem_fixed_count += 1
					for id in m_ALC_callback_ids do
						callbacks.removeScripts id:id
				)
				
				-- remove the globals
				if found_ALC_callbacks do
				(
					m_problem_fixed_count += 1
					for g in m_ALC_globals_to_clean do 
					(
						if (persistents.isPersistent g) do
							persistents.remove g
						globalVars.remove g
					)
				)
				
				-- replace the script controllera with something benign
				if found_ALC_script_controllers do with undo off
				(
					m_problem_fixed_count += 1
					for c in the_ALC_script_controllers do
					(
						c.SetExpression "[1,1,1]" -- so we don't detect controller as corruption even though just referenced by mxs value until next gc
						local new_c = bezier_scale()
						replaceinstances c new_c
					)
					the_ALC_script_controllers = undefined
				)
				
				setSaveRequired true
				if securitytools != undefined do
					securitytools.LogEventById (if doingPreSaveCheck then #SceneSaveALC else #SceneLoadALC) #CleanupSucceeded

				-- if m_auto_accept_next_process_clean is true, no messaging - using response from finding corruption in test_for_ALC_scripted_controller_callback
				if not m_auto_accept_next_process_clean do
				(
					if doingPreSaveCheck then
						msg = ~ALC_SECURITY_TOOL~ + m_version_number + ~ALC_SCENE_CLEAN_PERFORMED_PRESAVE~ + scene_file_name
					else
						msg = ~ALC_SECURITY_TOOL~ + m_version_number + ~ALC_SCENE_CLEAN_PERFORMED~ + scene_file_name
					logsystem.logentry msg info:true broadcast:true
					if display_listener_output do
						format "%\n" msg
					
					if not (m_force_local_quiet_mode or GetQuietMode()) do
					(	
						SecurityToolsDialogs_instance.display_corruptionCleanedInScene_dialog m_security_tools_ALC_help_topic_id
					)
				)
			)
			else 
			(
				-- if m_auto_reject_next_process_clean is true, no messaging - using response from finding corruption in test_for_ALC_scripted_controller_callback
				if not m_auto_reject_next_process_clean do
				(
					if securitytools != undefined do
						securitytools.LogEventById (if doingPreSaveCheck then #SceneSaveALC else #SceneLoadALC) #CleanupDeclined
					if doingPreSaveCheck then
						msg = ~ALC_SECURITY_TOOL~ + m_version_number + ~ALC_SCENE_CLEAN_NOT_PERFORMED_PRESAVE~ + scene_file_name
					else
						msg = ~ALC_SECURITY_TOOL~ + m_version_number + ~ALC_SCENE_CLEAN_NOT_PERFORMED~ + scene_file_name
					logsystem.logentry msg warning:true broadcast:true
					if display_listener_output do
						format "%\n" msg
				)
			)
		)
		
		if found_ALC_global and not m_auto_reject_next_process_clean do
			test_for_ALC_startup_scripts()
		
		-- if merging xref objects, don't reset auto accept values until dealing with the outermost merge
		-- if opening file, don't reset until done loading file
		if not m_in_objectXrefMerge and not m_in_sceneXrefMerge and not m_in_file_open then
			m_auto_accept_next_process_clean = m_auto_reject_next_process_clean = false
		else if process_corruption_found do
		(
			if m_perform_ALC_process_cleanup then
			(
				m_auto_accept_next_process_clean = true
				m_auto_reject_next_process_clean = false
			)
			else
			(
				m_auto_accept_next_process_clean = false
				m_auto_reject_next_process_clean = true
			)
		)
		
		if m_debug_mode_on do
			format "test_for_ALC_process_corruption[out] m_in_file_open:% m_in_objectXrefMerge:% m_in_sceneXrefMerge:% m_perform_ALC_process_cleanup:% m_auto_accept_next_process_clean:% m_auto_reject_next_process_clean:%\n" m_in_file_open m_in_objectXrefMerge m_in_sceneXrefMerge m_perform_ALC_process_cleanup m_auto_accept_next_process_clean m_auto_reject_next_process_clean
	),

	fn test_for_ALC =
	(
		test_for_ALC_process_corruption()
		-- this script may run prior to the ALC startup script file, so globals would not yet be present. So we want to test for the ALC startup script files.
		test_for_ALC_startup_scripts() 
		callbacks.removeScripts id:#ALC_SecurityTool_startup
		ok
	),
	
	-- The test_for_ALC_scripted_controller_callback function is called in response to a #nodeLinked event. This handles the case of merging and loading object xrefs, where we need to 
	-- clean the scripted controllers before they are evaluated by the massFX's px_watcher.ms script which does a 'classof' call on the merged nodes in response to #sceneNodeAdded events. 
	-- We can't do a full process cleanup at this point as it is not safe to delete nodes at this point in the load process.
	fn test_for_ALC_scripted_controller_callback =
	(
		if m_debug_mode_on do
			format "test_for_ALC_scripted_controller_callback m_in_file_open:% m_in_objectXrefMerge:% m_in_sceneXrefMerge:% m_auto_accept_next_process_clean:% m_auto_reject_next_process_clean:%\n" m_in_file_open m_in_objectXrefMerge m_in_sceneXrefMerge m_auto_accept_next_process_clean m_auto_reject_next_process_clean
		
		callbacks.removeScripts id:#ALC_SecurityTool_postload_nodelinked
		m_nodelinked_callback_registered = false
	
		-- if we are in a file open we can be doing multiple file merges as object xrefs are loaded.  So register a callback so that at the beginning
		-- of the next file merge we register a #nodeLinked callback that calls this method
		if m_in_file_open do
			callbacks.addScript #filePreMerge "::ALC_SecurityTool.register_nodelinked_callback()" id:#ALC_SecurityTool_postload

		local scaleScripts = getclassinstances scale_script processAllAnimatables:true
		local the_ALC_script_controllers = #()
		for scaleScript in scaleScripts do 
		(
			local expr = scaleScript.GetExpression()
			if (findstring expr m_scale_script_expression_string) != undefined do
				append the_ALC_script_controllers scaleScript
		)
		
		m_perform_ALC_process_cleanup = false
		if the_ALC_script_controllers.count != 0 do
		(
			m_problem_found_count += 1
			local scene_file_name = maxFilePath + maxFileName
			if scene_file_name == "" do scene_file_name = ~ALC_SCENE_UNNAMED~
				
			local msg
			local display_listener_output = m_force_local_quiet_mode or GetQuietMode() or m_force_print_to_listener
		
			if not (m_auto_accept_next_process_clean or m_auto_reject_next_process_clean) do
			(
				msg = ~ALC_SECURITY_TOOL~ + m_version_number + ~ALC_SCENE_DETECTED_MSG~ + scene_file_name
				logsystem.logentry msg warning:true broadcast:true
				if display_listener_output do
					format "%\n" msg
			)

			if m_auto_accept_next_process_clean then
				m_perform_ALC_process_cleanup = true
			else if m_auto_reject_next_process_clean then
				m_perform_ALC_process_cleanup = false
			else if m_force_local_quiet_mode or GetQuietMode() then
			(
				m_perform_ALC_process_cleanup = m_perform_cleanup_quiet_mode
				if m_perform_ALC_process_cleanup then
				(
					msg = ~ALC_SECURITY_TOOL~ + m_version_number + ~ALC_SCENE_AUTO_CLEAN_QUIET~
				)
				else
				(
					msg = ~ALC_SECURITY_TOOL~ + m_version_number + ~ALC_SCENE_NOT_AUTO_CLEAN_QUIET~
				)
				logsystem.logentry msg info:true broadcast:true
				if display_listener_output do
					format "%\n" msg
			)
			else
			(
				-- even though we disable viewport redraw in the dialog open handler, there is at least one code path that doesn't check whether viewport redraws are disabled
				-- before redrawing the viewports. While dialog is up, replace the script controllers in the scene with different controllers, and then restore.
				local controller_remappings = #()
				for c in the_ALC_script_controllers do with undo off
				(
					local new_c = bezier_scale()
					replaceinstances c new_c
					append controller_remappings (datapair c new_c)
				)
				m_perform_ALC_process_cleanup = SecurityToolsDialogs_instance.display_corruptionFoundInScene_dialog m_security_tools_ALC_help_topic_id "ALC"
				for item in controller_remappings do with undo off
					replaceInstances item.v2 item.v1
				if not m_perform_ALC_process_cleanup do
				(
					SecurityToolsDialogs_instance.display_corruptionNotCleanedInScene_dialog m_security_tools_ALC_help_topic_id
				)
			)
			
			if m_perform_ALC_process_cleanup then
			(
				m_problem_fixed_count += 1
				
				for c in the_ALC_script_controllers do with undo off 
				(
					c.SetExpression "[1,1,1]" -- so we don't detect controller as corruption even though just referenced by mxs value until next gc
					local new_c = bezier_scale()
					replaceinstances c new_c
				)
				the_ALC_script_controllers = undefined
				
				if not m_auto_accept_next_process_clean do
				(
					setSaveRequired true
					if securitytools != undefined do
						securitytools.LogEventById #SceneLoadALC #CleanupSucceeded
					
					msg = ~ALC_SECURITY_TOOL~ + m_version_number + ~ALC_SCENE_CLEAN_PERFORMED~ + scene_file_name
					logsystem.logentry msg info:true broadcast:true
					if display_listener_output do
						format "%\n" msg
					if not (m_force_local_quiet_mode or GetQuietMode()) then
					(	
						SecurityToolsDialogs_instance.display_corruptionCleanedInScene_dialog m_security_tools_ALC_help_topic_id
					)
					m_auto_accept_next_process_clean = true 
				)
			)
			else
			(
				if not m_auto_reject_next_process_clean do
				(
					if securitytools != undefined do
						securitytools.LogEventById #SceneLoadALC #CleanupDeclined
					msg = ~ALC_SECURITY_TOOL~ + m_version_number + ~ALC_SCENE_CLEAN_NOT_PERFORMED~ + scene_file_name
					logsystem.logentry msg warning:true broadcast:true
					if display_listener_output do
						format "%\n" msg
					m_auto_reject_next_process_clean = true
				)
			)
		)
		if m_debug_mode_on do
			format "test_for_ALC_scripted_controller_callback[out] m_in_file_open:% m_in_objectXrefMerge:% m_in_sceneXrefMerge:% m_auto_accept_next_process_clean:% m_auto_reject_next_process_clean:%\n" m_in_file_open m_in_objectXrefMerge m_in_sceneXrefMerge m_auto_accept_next_process_clean m_auto_reject_next_process_clean
	),

	-- flag variables for specifying that in the sceneXrefPostMerge and objectXrefPostMerge methods (called via corresponding event handlers) whether we should
	-- do call m_test_for_ALC_process_corruption_on_sceneXrefPostMerge. 
	m_test_for_ALC_process_corruption_on_sceneXrefPostMerge = false,
	m_test_for_ALC_process_corruption_on_objectXrefPostMerge = false, 
	
	fn register_nodelinked_callback =
	(
		if m_debug_mode_on do
			format "register_nodelinked_callback m_in_file_open:% m_in_objectXrefMerge:% m_in_objectXrefMerge_depth:% m_in_sceneXrefMerge:% m_in_sceneXrefMerge_depth:% m_nodelinked_callback_registered:%\n" m_in_file_open m_in_objectXrefMerge m_in_objectXrefMerge_depth m_in_sceneXrefMerge m_in_sceneXrefMerge_depth m_nodelinked_callback_registered
		if not m_nodelinked_callback_registered do
		(
			m_nodelinked_callback_registered = true
			callbacks.addScript #nodeLinked "::ALC_SecurityTool.test_for_ALC_scripted_controller_callback()" id:#ALC_SecurityTool_postload_nodelinked
		)
	),
	
	fn register_post_load_check =
	(
		if m_debug_mode_on do
			format "register_post_load_check m_in_file_open:% m_in_objectXrefMerge:% m_in_objectXrefMerge_depth:% m_in_sceneXrefMerge:% m_in_sceneXrefMerge_depth:% \n" m_in_file_open m_in_objectXrefMerge m_in_objectXrefMerge_depth m_in_sceneXrefMerge m_in_sceneXrefMerge_depth
		if not m_postloadCallbacks_registered do
		(
			m_postloadCallbacks_registered = true
			 -- #animationRangeChange notification fired after loading mxs persistents, but before the #filePostOpen notification is fired
			callbacks.addScript #animationRangeChange "::ALC_SecurityTool.test_for_ALC_process_corruption unregisterPostLoad:true" id:#ALC_SecurityTool_animationRangeChange
			callbacks.addScript #filePostOpen "::ALC_SecurityTool.m_in_file_open = false;::ALC_SecurityTool.test_for_ALC_process_corruption unregisterPostLoad:true" id:#ALC_SecurityTool_postload
			callbacks.addScript #fileOpenFailed "::ALC_SecurityTool.m_in_file_open = false;::ALC_SecurityTool.test_for_ALC_process_corruption unregisterPostLoad:true" id:#ALC_SecurityTool_postload
			register_nodelinked_callback()
			if not m_in_file_open do
			(
				if m_in_sceneXrefMerge then
					m_test_for_ALC_process_corruption_on_sceneXrefPostMerge = true
				else if m_in_objectXrefMerge then
					m_test_for_ALC_process_corruption_on_objectXrefPostMerge = true
				else
					callbacks.addScript #filePostMerge "::ALC_SecurityTool.test_for_ALC_process_corruption unregisterPostLoad:true useSafeNodeNameCollect:true" id:#ALC_SecurityTool_postload
			)
		)
	),
	
	fn maybe_pre_save_check =
	(
		local notificationParam = callbacks.notificationParam()
		if notificationParam[1] != 3 do -- 3 == autobackup
			test_for_ALC_process_corruption doingPreSaveCheck:true saveFileName:(notificationParam[2])
	),
	
	fn objectXrefPreMerge =
	(
		callbacks.addScript #filePreMerge "::ALC_SecurityTool.register_nodelinked_callback()" id:#ALC_SecurityTool_postload
		m_in_objectXrefMerge = true
		m_in_objectXrefMerge_depth += 1
	),
	
	fn objectXrefPostMerge =
	(
		if m_in_objectXrefMerge_depth > 0 do 
			m_in_objectXrefMerge_depth -= 1
		m_in_objectXrefMerge = m_in_objectXrefMerge_depth != 0
		if m_test_for_ALC_process_corruption_on_objectXrefPostMerge do
		(
			m_test_for_ALC_process_corruption_on_objectXrefPostMerge = false
			test_for_ALC_process_corruption unregisterPostLoad:true useSafeNodeNameCollect:true
		)
		if not m_in_objectXrefMerge and not m_in_sceneXrefMerge and not m_in_file_open then
			m_auto_accept_next_process_clean = m_auto_reject_next_process_clean = false
	),
	
	fn sceneXrefPreMerge =
	(
		m_in_sceneXrefMerge = true
		m_in_sceneXrefMerge_depth += 1
	),
	
	fn sceneXrefPostMerge =
	(
		if m_in_sceneXrefMerge_depth > 0 do 
			m_in_sceneXrefMerge_depth -= 1
		m_in_sceneXrefMerge = m_in_sceneXrefMerge_depth != 0
		if m_test_for_ALC_process_corruption_on_sceneXrefPostMerge do
		(
			m_test_for_ALC_process_corruption_on_sceneXrefPostMerge = false
			test_for_ALC_process_corruption unregisterPostLoad:true useSafeNodeNameCollect:true
		)
		if not m_in_objectXrefMerge and not m_in_sceneXrefMerge and not m_in_file_open then
			m_auto_accept_next_process_clean = m_auto_reject_next_process_clean = false
	),
	
	fn scenePreOpen =
	(
		m_in_file_open = true
		register_post_load_check()
	),
	
	fn scenePostOpen =
	(
		if m_debug_mode_on do
			format "scenePostOpen m_in_file_open:% m_auto_accept_next_process_clean:% m_auto_reject_next_process_clean:%\n" m_in_file_open m_auto_accept_next_process_clean m_auto_reject_next_process_clean
		m_in_file_open = false
		m_auto_accept_next_process_clean = false
		m_auto_reject_next_process_clean = false
	),
	
	on create do
	(
		::ALC_SecurityTool = this
		
		local msg = ~ALC_SECURITY_TOOL~ + m_version_number + ~ALC_SECURITY_TOOLS_LOADED~ + "  (" + getThisScriptFilename()  + ")"
		logsystem.logEntry msg info:true broadcast:true
		if m_force_local_quiet_mode or GetQuietMode() or m_force_print_to_listener do
			format "-- %\n" msg
		
		if (isproperty systemTools #GetBuildNumber) do
		(
			local buildNumberVals= filterstring (systemTools.GetBuildNumber()) "." --> #("20", "4", "0", "0")
			for i = 1 to buildNumberVals.count do buildNumberVals[i] = buildNumberVals[i] as integer
			m_callbacks_show_support_to = buildNumberVals[1] > 20
			if not m_callbacks_show_support_to do
				m_callbacks_show_support_to = buildNumberVals[1] == 20 and (buildNumberVals[4] > 4250 or buildNumberVals[4] == 0)
		)
		
		test_for_ALC()

		if m_debug_mode_on then
			enable_Debug()
		else
			disable_Debug()
		
		callbacks.removeScripts id:#ALC_SecurityTool
		callbacks.removeScripts id:#ALC_SecurityTool_startup
		callbacks.removeScripts id:#ALC_SecurityTool_postload
		callbacks.removeScripts id:#ALC_SecurityTool_postload_nodelinked
		callbacks.addScript #mtlLibPostOpen "::ALC_SecurityTool.test_for_ALC_process_corruption unregisterPostStartup:true" id:#ALC_SecurityTool_startup -- ensures run at least once at startup after all startup scripts are loaded
		callbacks.addScript #filePreOpen "::ALC_SecurityTool.scenePreOpen()" id:#ALC_SecurityTool
		callbacks.addScript #filePostOpen "::ALC_SecurityTool.scenePostOpen()" id:#ALC_SecurityTool
		callbacks.addScript #fileOpenFailed "::ALC_SecurityTool.scenePostOpen()" id:#ALC_SecurityTool
		callbacks.addScript #filePreMerge "::ALC_SecurityTool.register_post_load_check()" id:#ALC_SecurityTool
		callbacks.addScript #filePreSaveProcess "::ALC_SecurityTool.maybe_pre_save_check()" id:#ALC_SecurityTool
		callbacks.addScript #objectXrefPreMerge "::ALC_SecurityTool.objectXrefPreMerge()" id:#ALC_SecurityTool
		callbacks.addScript #objectXrefPostMerge "::ALC_SecurityTool.objectXrefPostMerge()" id:#ALC_SecurityTool
		callbacks.addScript #sceneXrefPreMerge "::ALC_SecurityTool.sceneXrefPreMerge()" id:#ALC_SecurityTool
		callbacks.addScript #sceneXrefPostMerge "::ALC_SecurityTool.sceneXrefPostMerge()" id:#ALC_SecurityTool
	)
)

(
	ALC_SecurityTool_structdef()
	silentValue
)

-------BEGIN-SIGNATURE-----
-- 4wYAADCCBt8GCSqGSIb3DQEHAqCCBtAwggbMAgEBMQ8wDQYJKoZIhvcNAQELBQAw
-- CwYJKoZIhvcNAQcBoIIE3jCCBNowggPCoAMCAQICEDUAFkMQxqI9PltZ2eUG16Ew
-- DQYJKoZIhvcNAQELBQAwgYQxCzAJBgNVBAYTAlVTMR0wGwYDVQQKExRTeW1hbnRl
-- YyBDb3Jwb3JhdGlvbjEfMB0GA1UECxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29yazE1
-- MDMGA1UEAxMsU3ltYW50ZWMgQ2xhc3MgMyBTSEEyNTYgQ29kZSBTaWduaW5nIENB
-- IC0gRzIwHhcNMTkwNjI1MDAwMDAwWhcNMjAwODA3MjM1OTU5WjCBijELMAkGA1UE
-- BhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExEzARBgNVBAcMClNhbiBSYWZhZWwx
-- FzAVBgNVBAoMDkF1dG9kZXNrLCBJbmMuMR8wHQYDVQQLDBZEZXNpZ24gU29sdXRp
-- b25zIEdyb3VwMRcwFQYDVQQDDA5BdXRvZGVzaywgSW5jLjCCASIwDQYJKoZIhvcN
-- AQEBBQADggEPADCCAQoCggEBAMsptjSEm+HPve6+DClr+K4CgrtrONjtHxHBwTMC
-- mrwF9bnsdMiSgvYigTKk858TlqVs7GiBVLD3SaSZqfSXOv7L55i965L+wIx0EZxX
-- xDzbyLh1rLSSNWO8oTDIKnPsiwo5x7CHRUi/eAICOvLmz7Rzi+becd1j/JPNWe5t
-- vum0GL/8G4vYICrhCycizGIuv3QFqv0YPM75Pd2NP0V4W87XPeTrj+qQoRKMztJ4
-- WNDgLgT4LbMBIZyluU8iwXNyWQ8FC2ya3iJyy0EhZhAB2H7oMrAcV1VJJqwZcZQU
-- XMJTD+tuCqKqJ1ftv1f0JVW2AADnHgvaB6E6Y9yR/jnn4zECAwEAAaOCAT4wggE6
-- MAkGA1UdEwQCMAAwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQMMAoGCCsGAQUFBwMD
-- MGEGA1UdIARaMFgwVgYGZ4EMAQQBMEwwIwYIKwYBBQUHAgEWF2h0dHBzOi8vZC5z
-- eW1jYi5jb20vY3BzMCUGCCsGAQUFBwICMBkMF2h0dHBzOi8vZC5zeW1jYi5jb20v
-- cnBhMB8GA1UdIwQYMBaAFNTABiJJ6zlL3ZPiXKG4R3YJcgNYMCsGA1UdHwQkMCIw
-- IKAeoByGGmh0dHA6Ly9yYi5zeW1jYi5jb20vcmIuY3JsMFcGCCsGAQUFBwEBBEsw
-- STAfBggrBgEFBQcwAYYTaHR0cDovL3JiLnN5bWNkLmNvbTAmBggrBgEFBQcwAoYa
-- aHR0cDovL3JiLnN5bWNiLmNvbS9yYi5jcnQwDQYJKoZIhvcNAQELBQADggEBADo7
-- 6cASiVbzkjsADk5MsC3++cj9EjWeiuq+zzKbe55p6jBNphsqLUvMw+Z9r2MpxTEs
-- c//MNUXidFsslWvWAUeOdtytNfhdyXfENX3baBPWHhW1zvbOPHQLyz8LmR1bNe9f
-- R1SLAezJaGzeuaY/Cog32Jh4qDyLSzx87tRUJI2Ro5BLA5+ELiY21SDZ7CP9ptbU
-- CDROdHY5jk/WeNh+3gLHeikJSM9/FPszQwVc9mjbVEW0PSl1cCLYEXu4T0o09ejX
-- NaQPg10POH7FequNcKw50L63feYRStDf6GlO4kNXKFHIy+LPdLaSdCQL2/oi3edV
-- MdpL4F7yw1zQBzShYMoxggHFMIIBwQIBATCBmTCBhDELMAkGA1UEBhMCVVMxHTAb
-- BgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8wHQYDVQQLExZTeW1hbnRlYyBU
-- cnVzdCBOZXR3b3JrMTUwMwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
-- b2RlIFNpZ25pbmcgQ0EgLSBHMgIQNQAWQxDGoj0+W1nZ5QbXoTANBgkqhkiG9w0B
-- AQsFADANBgkqhkiG9w0BAQEFAASCAQAGIFzHHFsfMM2Y0EwZYmBMpcEUkotNOK5e
-- CkJ/wpkOC/2og3Vr8tBBFTi2FPEYhwuuCDq7ng9YGxdo8231xq5YstWvFwBaYpBY
-- C4NkoFgWwQuM0wxtSj4L+4JSxGUDBealyBgcWcMoa1OqJe1CRDUVaJGV0GouZk1N
-- KvehgFr9oNvv4850iCTm/YQgwUqTtK7GuoYLxQSdUR1WG9VU6YJX8t25+9fp8pUk
-- QO7XDKvCjSrE6Wl3aiiF0fKZmOsnQD6P/kHaxtbIl3yFpQots2ER7y7uhR4h99b7
-- MzSHDKwODHszJYY+QnviF8n6TRMHhCMyRKsfTRcCdebzJkcEplZ4
-- -----END-SIGNATURE-----